//	TorusGamesGPUDefinitions.h
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#ifndef TorusGamesGPUDefinitions_h
#define TorusGamesGPUDefinitions_h

#include <simd/simd.h>

#define COLLAGE_MAX_JIGSAW_SIZE	5


enum
{
	VertexAttribute2DPosition	= 0,
	VertexAttribute2DTexCoords	= 1,
	VertexAttribute2DOffset		= 2
};
enum
{
	VertexAttribute3DWallPosition	= 0,
	VertexAttribute3DWallTexCoords	= 1,
	VertexAttribute3DWallWeight		= 2
};
enum
{
	VertexAttribute3DPolyhedronPosition	= 0,
	VertexAttribute3DPolyhedronNormal	= 1
};

//	Buffer indices for vertex functions ("VF")
enum
{
	BufferIndexVFVertexAttributes			= 0,
	BufferIndexVFWorldData					= 1,
	BufferIndexVFCoveringTransformations	= 2,
	BufferIndexVFPlacement					= 3,
	BufferIndexVFMisc						= 4,
	BufferIndexVFTexCoordShift				= 5
};

//	Buffer indices for fragment functions ("FF")
enum
{
	BufferIndexFFMisc	= 0
};

//	Buffer indices for compute functions ("CF")
enum
{
	BufferIndexCFMisc	= 0,
	BufferIndexCFColor	= 1
};

//	Texture and sampler indices for fragment functions ("FF")
enum
{
	TextureIndexFF	= 0
};
enum
{
	SamplerIndexFF	= 0
};

//	With one exception, each GPU compute function ("CF")
//	takes a single texture, which it uses for its output.
//	The exception is TorusGamesComputeFunctionMakeJigsawCollages(),
//	which takes as input both a source image and a piece template,
//	and produces as output two collages, one without piece borders
//	and one with.
enum
{
	TextureIndexCFImage	= 0
};
enum
{
	TextureIndexCFJigsawPuzzleSourceImage		= 0,
	TextureIndexCFJigsawPieceTemplate			= 1,
	TextureIndexCFJigsawCollageWithoutBorders	= 2,
	TextureIndexCFJigsawCollageWithBorders		= 3
};


//	2D

typedef struct
{
	simd_float3x3	itsDragPlacement;
	float			itsClippingDistance;
} TorusGames2DWorldData;

typedef simd_float3x3	TorusGames2DCoveringTransformation;
typedef simd_float3x3	TorusGames2DSpritePlacementMatrix;

#define MAX_MAZE_CELLS_PER_ROW	32
typedef struct
{
	ushort	itsCellsPerRow,
			itsPixelsPerCell,
			itsLineHalfWidth;
	bool	itsKleinBottleFlag;
	
	//	Each "wall", in its four lowest-order bits,
	//	contains a packed array of four 1-bit booleans
	//
	//		(00000000)0000WSEN
	//
	//	which tell whether a wall is present on the given cell's
	//	west, south, east or north side, respectively.
	//
	//	All the information in itsWalls[row][col] in contained
	//	in those four lowest-order bits, so in principle
	//	itsWalls could be of type uint8_t.  In practice
	//
	//	  - Defining theWalls as uint8_t works fine
	//		on my iPod Touch 6th gen (with an A8 system-on-a-chip),
	//		on my iPad Air 2 (with an A8x) and
	//		on my iPhone SE 2nd gen (with an A13).
	//
	//	  - Defining theWalls as uint8_t can be made to work
	//		on my iPhone 5s (with an A7 system-on-a-chip),
	//		if the GPU function immediately converts the uint8_t
	//		to a 32-bit unsigned integer in the line
	//
	//			theWalls = someParmeters.itsWalls[theCell.x][theCell.y];
	//
	//	  - Defining theWalls as uint8_t fails on Alex Jacoby's son's
	//		iPad 7th gen (with an A10), Adam's iPhone SE 1st gen (with an A9)
	//		and Nadia's iPhone 6s (also with an A9), whether or not
	//		the GPU function converts the uint8_t to a larger unsigned integer.
	//
	//	So let's leave itsWalls as a 16-bit ushort, for robustness.
	//	The total amount of memory is only 32*32*2 = 2048 bytes,
	//	which is almost negligible in the grand scheme of things.
	//
	ushort	itsWalls[MAX_MAZE_CELLS_PER_ROW][MAX_MAZE_CELLS_PER_ROW];	//	indexed as [row][col]

} MazeMaskParameters;

typedef struct
{
	//	For a full explanation of these layout parameters
	//	and how they're used, please see the comments
	//	and drawings in
	//
	//		-computeJigsawCollageLayout: ...
	//
	ushort			n,						//	typically ∈ {3, 4, 5}
					s,						//	typically 1024 (0x0400) or 2048 (0x0800)
					itsCoordinateMask,		//	= s - 1, typically 0x03FF or 0x07FF
					itsSubsequentOffsetPx,	//	= s / n (fractional part intentionally discarded)
					itsInitialOffsetPx,		//	= itsSubsequentOffsetPx / 2 (fractional part intentionally discarded)
					itsApproxBlockStride;	//	= (2s + (n-1))/n (fractional part intentionally discarded)
	simd_ushort2	itsPhaseShift;			//	∈ {0, …, s-1} × {0, …, s-1}
	bool			itsKleinBottleFlag;
	simd_float2		itsPieceWidthsPerPixel;	//	= n / s (as float)
	simd_ushort2	itsPieceTemplateSize;	//	= (1024, 512)
	simd_uchar4		itsPieceTabs[COLLAGE_MAX_JIGSAW_SIZE][COLLAGE_MAX_JIGSAW_SIZE];
						//	Each entry is a "bool4" saying whether there's a tab
						//	on the piece's (west, east, south, north) side.
						//	Note that the indexing [col][row] is in column-major order.
} JigsawCollageLayout;


//	3D

typedef struct
{
	//	for the scene as a whole
	//
	//		For an explanation of the various mappings, see "TorusGames-Common.h".
	//		Search for "In a 3D game, a sequence of maps".
	//

	simd_float4x4	itsTilingIntoFrameCell,	//	for ViewBasicLarge
					itsFrameCellIntoWorld,	//	for ViewBasicLarge
					itsTilingIntoWorld,		//	for ViewRepeating
					itsProjection;			//	maps world space to clip space
	
	float			itsClippingRadiusForRepeatingView;

	__fp16		//	itsFogBegin,	//	in world coordinates, not used explicity used in GPU fog computation
					itsFogEnd,		//	in world coordinates
					itsFogScale;	//	precompute 1.0 / (itsFogEnd - itsFogBegin) to avoid division at run time

	//	for frame cell walls
	simd_float2		itsAperture;	//	(a, 1 - a);  (0,1) = fully closed, (1,0) = fully open

} TorusGames3DWorldData;

typedef simd_float4x4	TorusGames3DCoveringTransformation;

typedef struct
{
	//	The shader applies itsDilation only to vertices, never to normal vectors.
	simd_float3		itsDilation;				//	scale factor in x, y and z directions
	
	//	The shader applies itsIsometricPlacement to normal vectors as well as vertices.
	simd_float4x4	itsIsometricPlacement;		//	reflects, rotates and translates as desired
	
	//	The shader applies itsExtraClippingCovector only in 3D Tic-Tac-Toe
	//	and even there only for cross sections of win-line tubes that intersect
	//	a frame cell wall obliquely.  Because the win-line tube has a finite length,
	//	sometimes its cross section is a clipped ellipse rather than a full ellipse.
	//	The shader evaluates the covector in frame cell coordinates
	//	and clips to the region where the covector times a vertex
	//	evaluates to a result in the range [-1.0, +1.0].
	simd_float4		itsExtraClippingCovector;	//	to be evaluated on a point (x,y,z,1) in frame cell coordinates

} TorusGames3DPolyhedronPlacementAsSIMD;


#endif	//	TorusGamesGPUDefinitions_h
